#!/bin/bash

#-----------------------------------------------------------------------------
#  Copyright (c) 2017-2019 by Dell Inc.
#
# All rights reserved.  
#
#-----------------------------------------------------------------------------


#
# Get the Operating System
#
OS=`uname`;

scriptVersion="1.1.0";

echo "";
echo "------------------------------------------------------";
echo "Live Optics collection script for NetWorker on $OS";
echo "------------------------------------------------------";
date;
echo "Version: $scriptVersion";
echo "";

#
# Invariant formating and english language
#
LC_ALL=C;
export LC_ALL;
LANG=C;
export LANG;

#
# Paths
#
PATH=$PATH:/usr/sbin
export PATH;

SUFFIX=`date '+%d%m%Y%H%M%S'`;
OUTFOLDER="./LiveOpticsNetWorker"${SUFFIX};
CMP=gzip
PREFIX=nwquick

#
# Create the temp folder
#
echo "Making temporary folder for file output: $OUTFOLDER";
mkdir $OUTFOLDER;
set -x

#
# Check the disk space in the output folder.
#
DISK_SPACE_KB=0;

if [ $OS = 'SunOS' ]; then

	DISK_SPACE_KB=`df -b $OUTFOLDER | awk 'NR==2 { print $2 }'`;
	echo "$DISK_SPACE_KB KB available.";
	
elif [ $OS = 'Linux' ]; then

	DISK_SPACE_KB=`df -P $OUTFOLDER | awk '{if (NR!=1) { print $4 } }'`;
	DISK_SPACE_HUMAN=`df -Ph $OUTFOLDER | awk '{if (NR!=1) { print $4 } }'`;
	echo "$DISK_SPACE_HUMAN available.";

fi;

if [ "$DISK_SPACE_KB" -lt "4194304" ]; then
	echo "At least 4 GB of free disk space is recommended.";
	echo "Do you wish to proceed? (y/n) [n]:";
	read PROCEED;
	if [ "$PROCEED" = "y" -o "$PROCEED" = "Y" ]; then
		echo "Proceeding";
	else
		exit;
	fi;
fi;

#
# Write the local time.
#
date > "$OUTFOLDER"/"$PREFIX"_date.out;

#
# Script version
#
echo $scriptVersion > "$OUTFOLDER"/"$PREFIX"_scriptVersion.out;

#
# Base date used for drawing the line on backups.
# 120 days is the default.
#
lastBackupDate=$(
    perl -e 'use POSIX qw(strftime); \
    	print strftime "%m/%d/%Y", localtime(time()- 120 * 24 * 60 * 60);'
);

lastBackupDateString="-t $lastBackupDate";
echo "Last backup of interest: $lastBackupDate";
echo $lastBackupDateString > "$OUTFOLDER"/"$PREFIX"_lastBackupDateString.out

#
# Grab a bunch of system settings.
#
set > "$OUTFOLDER"/"$PREFIX"_set.out
uname -a > "$OUTFOLDER"/"$PREFIX"_uname.out
hostname > "$OUTFOLDER"/"$PREFIX"_hostname.out

#
# Set the Command file for nsradmin
#
COMMANDFILE="$OUTFOLDER"/commandfile;

echo "show version" > "$COMMANDFILE";
echo "p NSR" >> "$COMMANDFILE";

echo "Running command: nsradmin -i";
nsradmin -i "$COMMANDFILE" 2>"$OUTFOLDER"/"$PREFIX"_nsradminversion_err.out > "$OUTFOLDER"/"$PREFIX"_nsradminversion.out
if [ $? -ne 0 ] ; then
	echo "ERROR.  Failed to execute the nsradmin tool.";
	cat "$OUTFOLDER"/"$PREFIX"_nsradminversion_err.out;
	exit 1			
fi;

NETWORKER_VERSION=`cat "$OUTFOLDER"/"$PREFIX"_nsradminversion.out | \
	awk -F'NetWorker [^0-9]*' '$0=$2' \
	2> "$OUTFOLDER"/"$PREFIX"_nsradmin_awk_version_err.out` 

#
# Parse the major and minor versions of the version string.
#
NETWORKER_MAJOR_VERSION=`echo $NETWORKER_VERSION | awk '{split($0, major, "."); print major[1]}'`
NETWORKER_MINOR_VERSION=`echo $NETWORKER_VERSION | awk '{split($0, minor, "."); print minor[2]}'`


# 
# If parsing the version numbers fails, assume this is 7.4
#
if expr $NETWORKER_MAJOR_VERSION + 0 > /dev/null 2>&1; then 
	echo "NetWorker Major version: $NETWORKER_MAJOR_VERSION";
	echo "NetWorker Minor version: $NETWORKER_MINOR_VERSION";
else
	NETWORKER_MAJOR_VERSION=7;
	NETWORKER_MINOR_VERSION=4;
	echo "Assuming NetWorker Major version: $NETWORKER_MAJOR_VERSION";
	echo "Assuming NetWorker Major version: $NETWORKER_MINOR_VERSION";
fi;

if expr $NETWORKER_MINOR_VERSION + 0 > /dev/null 2>&1; then 
	NETWORKER_MINOR_VERSION=0;
fi;


# If version is >= 9.2 *or* 18.x, use ISO flag:
isoTimestampFlag="";
if [ "$NETWORKER_MAJOR_VERSION" -eq 9 -a "$NETWORKER_MINOR_VERSION" -ge 2 ] || [ "$NETWORKER_MAJOR_VERSION" -ge 18 ]; then
	isoTimestampFlag="-I";
fi;

echo "";
echo "isoTimestampFlag = $isoTimestampFlag";


echo show > "$COMMANDFILE";
echo option hidden >> "$COMMANDFILE";
echo p >> "$COMMANDFILE";

echo "Running command nsradmin -i";
nsradmin -i "$COMMANDFILE" > "$OUTFOLDER"/"$PREFIX"_nsradmin.out


echo "Running command srlic";
nsrlic -v 2>"$OUTFOLDER"/"$PREFIX"_nsrlic_err.out >  "$OUTFOLDER"/"$PREFIX"_nsrlic.out 

echo "Running command mminfo $isoTimestampFlag -av";

if [ "$NETWORKER_MAJOR_VERSION" -lt 7 ] || [ "$NETWORKER_MAJOR_VERSION" -eq 7 -a "$NETWORKER_MINOR_VERSION" -lt 5 ]; then
   mminfo $isoTimestampFlag -av -xm $lastBackupDateString -r 'ssid,state,volume,client,pool,capacity,location,name,sscreate(50),sscomp(50),savetime(50),level,totalsize,nfiles,ssretent(50),copies,ssflags,group,ssbrowse(50),cloneid,clonetime(50),clflags,fragsize,sumsize,ssaccess(50),attrs' \
		2>"$OUTFOLDER"/"$PREFIX"_mminfojobs_err.out | $CMP > "$OUTFOLDER"/"$PREFIX"_mminfojobs.out.gz
elif  [ "$NETWORKER_MAJOR_VERSION" -eq 7 -a "$NETWORKER_MINOR_VERSION" -ge 5 ] || [ "$NETWORKER_MAJOR_VERSION" -eq 8 -a "$NETWORKER_MINOR_VERSION" -lt 1 ]; then 
   mminfo $isoTimestampFlag -av -xm $lastBackupDateString -r 'ssid,state,volume,client,pool,capacity,location,name,sscreate(50),sscomp(50),savetime(50),level,totalsize,nfiles,ssretent(50),copies,ssflags,group,ssbrowse(50),cloneid,clonetime(50),clflags,fragsize,sumsize,ssaccess(50),clretent(50),attrs' \
		2>"$OUTFOLDER"/"$PREFIX"_mminfojobs_err.out | $CMP > "$OUTFOLDER"/"$PREFIX"_mminfojobs.out.gz
else 
   mminfo $isoTimestampFlag -av -xm $lastBackupDateString -r 'ssid,state,volume,client,pool,capacity,location,name,sscreate(50),sscomp(50),savetime(50),level,totalsize,nfiles,ssretent(50),copies,ssflags,group,ssbrowse(50),cloneid,clonetime(50),clflags,fragsize,sumsize,ssaccess(50),clretent(50),vmname,backup_size,attrs' 2>"$OUTFOLDER"/"$PREFIX"_mminfojobs_err.out | $CMP > "$OUTFOLDER"/"$PREFIX"_mminfojobs.out.gz
fi;

if [ ! -s "$OUTFOLDER"/"$PREFIX"_mminfojobs.out.gz ] ; then
	echo "ERROR: There was an error running mminfo to get the jobs."; 
	cat "$OUTFOLDER"/"$PREFIX"_mminfojobs_err.out 
	exit 1;
fi;

mminfo $isoTimestampFlag -av -xm $lastBackupDateString -r 'ssid,ssid(53)' \
	2>"$OUTFOLDER"/"$PREFIX"_mminfojobs_long.err | $CMP > "$OUTFOLDER"/"$PREFIX"_mminfojobs_long.out.gz 

echo "Running command mminfo $isoTimestampFlag -avV";
mminfo $isoTimestampFlag -av -xm -r 'state,volume,written,pool,%used,read,capacity,avail,volflags,type,family,barcode,volid,location,volaccess,volretent(50),olabel,labeled,recycled,volattrs' \
	2>"$OUTFOLDER"/"$PREFIX"_mminfomedia_err.out | $CMP > "$OUTFOLDER"/"$PREFIX"_mminfomedia.out.gz
	
if [ ! -s "$OUTFOLDER"/"$PREFIX"_mminfomedia.out.gz ] ; then
	echo "ERROR: There was an error running mminfo to get the jobs."; 
	cat "$OUTFOLDER"/"$PREFIX"_mminfojobs_err.out 
	exit 1;
fi;

echo "Running command mminfo $isoTimestampFlag -X -a -xm";
mminfo $isoTimestampFlag -X -a -xm 2>"$OUTFOLDER"/"$PREFIX"_mminfosummary.out > "$OUTFOLDER"/"$PREFIX"_mminfosummary_err.out


#
# Mark how long that took.
#
date > "$OUTFOLDER"/"$PREFIX"_date2.out

f_nsradmin_single()
{
	METHOD="$1";
 
	echo "Running command nsradmin -i $METHOD";
	echo "show" > "$COMMANDFILE";
	echo "option hidden" >> "$COMMANDFILE";
	echo ". NSR $METHOD" >> "$COMMANDFILE";
	echo "p" >> "$COMMANDFILE";
	nsradmin -i "$COMMANDFILE" 2>>"$OUTFOLDER"/"$PREFIX"_nsradmin_single_err.out >> "$OUTFOLDER"/"$PREFIX"_nsradmin_single.out
	echo "" >> "$OUTFOLDER"/"$PREFIX"_nsradmin_single.out
}

f_nsradmin_single;
f_nsradmin_single "client";
f_nsradmin_single "pool";


date > "$OUTFOLDER"/"$PREFIX"_date_end.out;

#
# TARGZ the files
#
echo "Packaging and compressing files into tar.gz file...";
PACKAGE="$PREFIX"_`hostname`.tar;

tar -cvf $PACKAGE "$OUTFOLDER"/"$PREFIX"_*out*;
$CMP $PACKAGE;

echo "Done.  Please upload the ${PACKAGE}.gz file to Live Optics.";