--
--
-- script to generate AWR collections
-- for every interval
--
-- this script also generates a capacity report
-- ensure capacity_11g.sql, capacity_12c.sql, splunk_capacity_11g.sql and splunk_capacity_12c.sql
-- are in the current directory
--
-- usage: genawrshtml.sql [RANGE] - where range is the number of hours worth of data to collect
--
-- connect as any user with access to the AWR Data Dictionary views, such as the SYSTEM account.
--
set echo off
set heading off
set feedback off
set pagesize 50000
set linesize 999
set trimspool on

alter session set nls_date_language=american;
 
whenever sqlerror exit sql.sqlcode;

--
-- check to see if AWR has been used on this database
-- if not throw an exception as we dont want to violate the license
--
variable licuse       number;
variable retchar      char;

begin

  -- check license entitlement
  select max(detected_usages) into :licuse
  from dba_feature_usage_statistics
  where 1=1
  and dbid = (select dbid from v$database)
  and upper(name) like 'AUTOMATIC WORKLOAD REPOSITORY';

  if( :licuse <1 ) then
    dbms_output.put_line( 'AWR entitlement check' );
    -- on some database systems this check fails even when the option is licensed
    -- remove the following line ONLY if you are certain that you are licensed for the performance and diagnostic license
    raise_application_error(-20001,'No AWR usage found on this database - check license entitlement');
  end if;

end;
/

whenever sqlerror continue

undef begin_snap
undef end_snap

prompt
prompt
prompt
prompt Dell Snapshot Oracle Collection Tool
prompt ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
prompt
prompt WARNING: only use this script if you are licensed for
prompt the Tuning and Diagnostics pack.  If you are not licensed
prompt for the Tuning and Diagnostics pack, press CTRL-C now and
prompt use the Statspack option instead.
prompt

--
-- get the range from the command line
-- or prompt the user for it
--

variable snaprange    number;

prompt Enter the AWR snapshot range to collect in hours
define snap_range = '&1'

-- if no answer is given use 168 hours
begin

  select
    to_number(nvl('&snap_range',168))
  into :snaprange from dual;

end;
/

--
-- create a spool file
--
column ext     new_value new_ext
column dbname  new_value new_dbname

select name dbname, '.sql' ext from V$DATABASE;

spool awr_collect_&new_dbname&new_ext

select 'prompt collecting AWR data - this may take a while' from dual;
select 'set echo off'       from dual;
select 'set term off'       from dual;
select 'set heading off'    from dual;
select 'set feedback off'   from dual;
select 'set pagesize 50000' from dual;
select 'set linesize 999'   from dual;
select 'set trimspool on'   from dual;
select 'alter session set nls_date_language=american;' from dual;

--
-- generate the AWR reports
--
with summary_t as (
select
  dbid,
  begin_interval_time,
  end_interval_time,
  instance_number,
  snap_id,
  last_snap_id,
  startup_time,
  last_startup_time
from
  (
  select
    snap.dbid,
    snap.begin_interval_time,  
    snap.end_interval_time,
    snap.instance_number,
    snap.snap_id,
    lag(snap.snap_id,1)over(order by snap.instance_number,snap.snap_id) last_snap_id,
    snap.startup_time,
    lag(snap.startup_time,1)over(order by snap.instance_number,snap.snap_id) last_startup_time
  from
    dba_hist_snapshot snap,
    v$database vd
  where 1=1
    and snap.dbid = vd.dbid
    and snap.begin_interval_time > sysdate-(:snaprange/24)
  order by snap.snap_id
  )
  where startup_time = last_startup_time
)
select
  'set term on'||chr(10)||
  'prompt collecting '||rownum||' of '||(select to_char(count(*)) from summary_t)||chr(10)||
  'set term off'||chr(10)||
  'spool awrrpt_'||dbid||'_'||instance_number||'_'||last_snap_id||'_'||snap_id||'.html'||chr(10)||
  'select output from table ( dbms_workload_repository.awr_report_html('||
    dbid||','||
    instance_number||','||
    last_snap_id||','||
    snap_id||'));'||chr(10)||
  'spool off' "-- command"
from
  summary_t
/


--
-- generate the capacity reports
--
select 'set term on' "-- command" from dual
union all
select 'prompt collecting capacity data' from dual
union all
select 'set term off' from dual
--
union all
--
select '@@capacity_12c.sql'||chr(10)||'@@capacity_splunk_12c.sql' "-- generate capacity report"
from v$instance
where regexp_substr( version, '[^.]*', 1, 1, 'i') not in ('10','11')
--
union all
--
select '@@capacity_11g.sql'||chr(10)||'@@capacity_splunk_11g.sql'
from v$instance
where regexp_substr( version, '[^.]*', 1, 1, 'i') in( '11' )
--
union all
--
select '@@capacity_10g.sql'||chr(10)||'@@capacity_splunk_10g.sql'
from v$instance
where regexp_substr( version, '[^.]*', 1, 1, 'i') in( '10' )
/

--
-- notify that report is complete
--
select 'set term on' from dual;
select 'prompt AWR data collection complete' from dual;

spool off

prompt
prompt
--
--

--
-- now execute the script awr_collect_&new_dbname&new_ext
--
prompt now execute the script awr_collect_&new_dbname&new_ext
@@awr_collect_&new_dbname&new_ext

