#!/bin/sh

#-----------------------------------------------------------------------------
#  Copyright (c) 2017-2019 by Dell Inc.
#
# All rights reserved.  
#
#-----------------------------------------------------------------------------

#
# Get the Operating System
#
OS=`uname`;

scriptVersion="1.0.9";

echo "";
echo "------------------------------------------------------";
echo "Live Optics collection script for ";
echo "IBM Spectrum Protect (TSM) on $OS";
echo "Tested with TSM Version 8.1.5.0"
echo "------------------------------------------------------";
date;
echo "Version: $scriptVersion";
echo "";


#
# Invariant formating and english language
#
LC_ALL=C;
export LC_ALL;
LANG=C;
export LANG;

#
# Get local directory information on the server
#
echo "Enter your IBM Spectrum Protect Client directory: " 
read tsmdirectory

echo "Enter your IBM dsm.opt full path (including file name): " 
read dsmoptpath

echo "Either enter a ServerAddress or hit enter to use the default in the dsm.opt file: "
read serverAddress

#
# Paths
#
PATH=$PATH:$tsmdirectory
export PATH;

SUFFIX=`date '+%d%m%Y%H%M%S'`;
OUTFOLDER="./LO_TSM_"${SUFFIX};
CMP=gzip
PREFIX=LO_TSM

#
# Create the temp folder
#
echo "Making temporary folder for file output: $OUTFOLDER";
mkdir $OUTFOLDER;

#
# Check the disk space in the output folder.
#
DISK_SPACE_KB=0;

if [ $OS = 'SunOS' ]; then

	DISK_SPACE_KB=`df -b $OUTFOLDER | awk 'NR==2 { print $2 }'`;
	echo "$DISK_SPACE_KB KB available.";
	
elif [ $OS = 'Linux' ]; then

	DISK_SPACE_KB=`df -P $OUTFOLDER | awk '{if (NR!=1) { print $4 } }'`;
	DISK_SPACE_HUMAN=`df -Ph $OUTFOLDER | awk '{if (NR!=1) { print $4 } }'`;
	echo "$DISK_SPACE_HUMAN available.";

fi;

if [ "$DISK_SPACE_KB" -lt "4194304" ]; then
	echo "At least 4 GB of free disk space is recommended.";
	echo "Do you wish to proceed? (y/n) [n]:";
	read PROCEED;
	if [ "$PROCEED" = "y" -o "$PROCEED" = "Y" ]; then
		echo "Proceeding";
	else
		exit;
	fi;
fi;

#
# Write the local time.
#
date > "$OUTFOLDER"/"$PREFIX"_date.out;

#
# Grab a bunch of system settings.
#
set > "$OUTFOLDER"/"$PREFIX"_set.out
uname -a > "$OUTFOLDER"/"$PREFIX"_uname.out
hostname > "$OUTFOLDER"/"$PREFIX"_hostname.out

echo $scriptVersion > "$OUTFOLDER"/"$PREFIX"_scriptVersion.out;

#
# Create a list of tables to query from the Database
#
tableList="AR_COPYGROUPS BU_COPYGROUPS OCCUPANCY"

#
# Get program credentials
#
echo "Enter your IBM Spectrum Protect Username: " 
read tsmusername

echo "Enter your IBM Spectrum Protect Password: " 
#
# Turn off echo for security
#
stty -echo
read tsmpassword
stty echo

#If the serverAddress is empty, run the dsmadmc commands without the -se option
if [ -z "$serverAddress" ]; then
	for table in $tableList; do
		echo "Generating ${PREFIX}${table}.out";
		dsmadmc "-id=$tsmusername" "-pass=$tsmpassword" "-optfile=$dsmoptpath" "-dataonly=yes" "-DISPL=TAB" "-TAB" "select COLNAME from COLUMNS WHERE tabname = '$table' Order By COLNO"| tr "," "-" | tr "\\n" "," | tr -d '\r' | sed 's/.$//' > "$OUTFOLDER"/"$PREFIX"_$table.out
		printf "\\r\\n" >> "$OUTFOLDER"/"$PREFIX"_$table.out
		dsmadmc "-id=$tsmusername" "-pass=$tsmpassword" "-optfile=$dsmoptpath" "-dataonly=yes" "-DISPL=TAB" "-TAB" "select * from $table" | tr -d "," | tr "\\t" "," >> "$OUTFOLDER"/"$PREFIX"_$table.out 2>  "$OUTFOLDER"/"$PREFIX"_"$table"_err.out
	done;
	
	echo "Generating "$PREFIX"_NODES.out";
	dsmadmc "-id=$tsmusername" "-pass=$tsmpassword" "-optfile=$dsmoptpath" "-dataonly=yes" "-DISPL=TAB" "-TAB" "select COLNAME from COLUMNS WHERE tabname = 'NODES' Order By COLNO" | tr "," "-" | tr "\\n" "," | tr -d '\r' | sed 's/.$//' > "$OUTFOLDER"/"$PREFIX"_NODES.out
	printf "\\r\\n" >> "$OUTFOLDER"/"$PREFIX"_NODES.out
	dsmadmc "-id=$tsmusername" "-pass=$tsmpassword" "-optfile=$dsmoptpath" "-dataonly=yes" "-DISPL=TAB" "-TAB" "select * from NODES WHERE LASTACC_TIME >= (CURRENT DATE - 1 YEAR)" | tr "," "-" | tr "\\t" "," >> "$OUTFOLDER"/"$PREFIX"_NODES.out 2>  "$OUTFOLDER"/"$PREFIX"_NODES_err.out	

	echo "Generating "$PREFIX"_SUMMARY.out";
	dsmadmc "-id=$tsmusername" "-pass=$tsmpassword" "-optfile=$dsmoptpath" "-dataonly=yes" "-DISPL=TAB" "-TAB" "select COLNAME from COLUMNS WHERE tabname = 'SUMMARY' Order By COLNO" | tr "," "-" | tr "\\n" "," | tr -d '\r' | sed 's/.$//' > "$OUTFOLDER"/"$PREFIX"_SUMMARY.out
	printf "\\r\\n" >> "$OUTFOLDER"/"$PREFIX"_SUMMARY.out
	dsmadmc "-id=$tsmusername" "-pass=$tsmpassword" "-optfile=$dsmoptpath" "-dataonly=yes" "-DISPL=TAB" "-TAB" "select * from SUMMARY WHERE START_TIME >= (CURRENT DATE - 60 DAY) AND ACTIVITY IN ('BACKUP', 'ARCHIVE', 'INCR_DBBACKUP', 'FULL_DBBACKUP', 'RESTORE', 'RETRIEVE')" | tr "," "-" | tr "\\t" "," >> "$OUTFOLDER"/"$PREFIX"_SUMMARY.out 2>  "$OUTFOLDER"/"$PREFIX"_SUMMARY_err.out

	echo "Generating "$PREFIX"_FILESPACES.out";
	dsmadmc "-id=$tsmusername" "-pass=$tsmpassword" "-optfile=$dsmoptpath" "-dataonly=yes" "-DISPL=TAB" "-TAB" "select COLNAME from COLUMNS WHERE tabname = 'FILESPACES' Order By COLNO" | tr "," "-" | tr "\\n" "," | tr -d '\r' | sed 's/.$//' > "$OUTFOLDER"/"$PREFIX"_FILESPACES.out
	printf "\\r\\n" >> "$OUTFOLDER"/"$PREFIX"_FILESPACES.out
	dsmadmc "-id=$tsmusername" "-pass=$tsmpassword" "-optfile=$dsmoptpath" "-dataonly=yes" "-DISPL=TAB" "-TAB" "select * from FILESPACES WHERE BACKUP_START >= (CURRENT DATE - 60 DAY)" | tr "," "-" | tr "\\t" "," >> "$OUTFOLDER"/"$PREFIX"_FILESPACES.out 2>  "$OUTFOLDER"/"$PREFIX"_FILESPACES_err.out
	
	echo "Generating "$PREFIX"_Version.out";
	printf "Version, Release, Level, Sublevel"  > "$OUTFOLDER"/"$PREFIX"_Version.out
	printf "\\r\\n" >> "$OUTFOLDER"/"$PREFIX"_Version.out
	dsmadmc "-id=$tsmusername" "-pass=$tsmpassword" "-optfile=$dsmoptpath" "-dataonly=yes" "-DISPL=TAB" "-TAB" "select Version, Release, Level, Sublevel from STATUS" | tr "," "-" | tr "\\t" "," >> "$OUTFOLDER"/"$PREFIX"_Version.out 2>  "$OUTFOLDER"/"$PREFIX"_Version_err.out
	
else
	echo "ServerAddress: $serverAddress";
	for table in $tableList; do
		echo "Generating ${PREFIX}${table}.out";
		dsmadmc "-id=$tsmusername" "-pass=$tsmpassword" "-se=$serverAddress" "-optfile=$dsmoptpath" "-dataonly=yes" "-DISPL=TAB" "-TAB" "select COLNAME from COLUMNS WHERE tabname = '$table' Order By COLNO" |tr "," "-" | tr "\\n" "," | tr -d '\r'| sed 's/.$//' > "$OUTFOLDER"/"$PREFIX"_$table.out
		printf "\\r\\n" >> "$OUTFOLDER"/"$PREFIX"_$table.out
		dsmadmc "-id=$tsmusername" "-pass=$tsmpassword" "-se=$serverAddress" "-optfile=$dsmoptpath" "-dataonly=yes" "-DISPL=TAB" "-TAB" "select * from $table" | tr "," "-" | tr "\\t" "," >> "$OUTFOLDER"/"$PREFIX"_$table.out 2>  "$OUTFOLDER"/"$PREFIX"_"$table"_err.out
	done;

#
#Table queries filtering by date
#
	echo "Generating "$PREFIX"_NODES.out";
	dsmadmc "-id=$tsmusername" "-pass=$tsmpassword" "-se=$serverAddress" "-optfile=$dsmoptpath" "-dataonly=yes" "-DISPL=TAB" "-TAB" "select COLNAME from COLUMNS WHERE tabname = 'NODES' Order By COLNO" | tr "," "-" | tr "\\n" "," | tr -d '\r'| sed 's/.$//' > "$OUTFOLDER"/"$PREFIX"_NODES.out
	printf "\\r\\n" >> "$OUTFOLDER"/"$PREFIX"_NODES.out
	dsmadmc "-id=$tsmusername" "-pass=$tsmpassword" "-se=$serverAddress" "-optfile=$dsmoptpath" "-dataonly=yes" "-DISPL=TAB" "-TAB" "select * from NODES WHERE LASTACC_TIME >= (CURRENT DATE - 1 YEAR)" | tr "," "-" | tr "\\t" "," >> "$OUTFOLDER"/"$PREFIX"_NODES.out 2>  "$OUTFOLDER"/"$PREFIX"_NODES_err.out
	
	echo "Generating "$PREFIX"_SUMMARY.out";																																																																							  
	dsmadmc "-id=$tsmusername" "-pass=$tsmpassword" "-se=$serverAddress" "-optfile=$dsmoptpath" "-dataonly=yes" "-DISPL=TAB" "-TAB" "select COLNAME from COLUMNS WHERE tabname = 'SUMMARY' Order By COLNO" | tr "," "-" | tr "\\n" "," | tr -d '\r'| sed 's/.$//' > "$OUTFOLDER"/"$PREFIX"_SUMMARY.out
	printf "\\r\\n" >> "$OUTFOLDER"/"$PREFIX"_SUMMARY.out
	dsmadmc "-id=$tsmusername" "-pass=$tsmpassword" "-se=$serverAddress" "-optfile=$dsmoptpath" "-dataonly=yes" "-DISPL=TAB" "-TAB" "select * from SUMMARY WHERE START_TIME >= (CURRENT DATE - 60 DAY) AND ACTIVITY IN ('BACKUP', 'ARCHIVE', 'INCR_DBBACKUP', 'FULL_DBBACKUP', 'RESTORE', 'RETRIEVE')" | tr "," "-" | tr "\\t" "," >> "$OUTFOLDER"/"$PREFIX"_SUMMARY.out 2>  "$OUTFOLDER"/"$PREFIX"_SUMMARY_err.out

	echo "Generating "$PREFIX"_FILESPACES.out";
	dsmadmc "-id=$tsmusername" "-pass=$tsmpassword" "-se=$serverAddress" "-optfile=$dsmoptpath" "-dataonly=yes" "-DISPL=TAB" "-TAB" "select COLNAME from COLUMNS WHERE tabname = 'FILESPACES' Order By COLNO" | tr "," "-" | tr "\\n" ","| tr -d '\r' | sed 's/.$//' > "$OUTFOLDER"/"$PREFIX"_FILESPACES.out
	printf "\\r\\n" >> "$OUTFOLDER"/"$PREFIX"_FILESPACES.out
	dsmadmc "-id=$tsmusername" "-pass=$tsmpassword" "-se=$serverAddress" "-optfile=$dsmoptpath" "-dataonly=yes" "-DISPL=TAB" "-TAB" "select * from FILESPACES WHERE BACKUP_START >= (CURRENT DATE - 60 DAY)" | tr "," "-" | tr "\\t" "," >> "$OUTFOLDER"/"$PREFIX"_FILESPACES.out 2>  "$OUTFOLDER"/"$PREFIX"_FILESPACES_err.out
	
	echo "Generating "$PREFIX"_Version.out";
	printf "Version, Release, Level, Sublevel"  > "$OUTFOLDER"/"$PREFIX"_Version.out
	printf "\\r\\n" >> "$OUTFOLDER"/"$PREFIX"_Version.out
	dsmadmc "-id=$tsmusername" "-pass=$tsmpassword" "-optfile=$dsmoptpath" "-dataonly=yes" "-DISPL=TAB" "-TAB" "select Version, Release, Level, Sublevel from STATUS" | tr "," "-" | tr "\\t" "," >> "$OUTFOLDER"/"$PREFIX"_Version.out 2>  "$OUTFOLDER"/"$PREFIX"_Version_err.out

fi;

#
# Clean up
#
unset $tsmusername
unset $tsmpassword

#
# Done.  Mark our final timestamp.
#
date > "$OUTFOLDER"/"$PREFIX"_date2.out

#
# TARGZ the files
#
echo "Packaging and compressing files into tar.gz file...";
PACKAGE="$PREFIX"_`hostname`.tar;

tar -cvf $PACKAGE "$OUTFOLDER"/"$PREFIX"_*out*;
$CMP $PACKAGE;

echo "  "
echo "Done.  Please upload the ${PACKAGE}.gz file to Live Optics.";



