#!/bin/sh 

#-----------------------------------------------------------------------------
#  Copyright (c) 2017-2018 by Dell Inc.
#
# All rights reserved.  
#
#-----------------------------------------------------------------------------


#
# Get the Operating System
#
OS=`uname`;

scriptVersion="1.0.009";

echo "";
echo "------------------------------------------------------";
echo "Live Optics collection script for NetBackup on $OS";
echo "------------------------------------------------------";
date;
echo "Version: $scriptVersion";
echo "";

#
# Invariant formating and english language
#
LC_ALL=C;
export LC_ALL;
LANG=C;
export LANG;

#
# Paths
#
OPENV=/usr/openv
PATH=$PATH:$OPENV/volmgr/bin:$OPENV/netbackup/bin:$OPENV/netbackup/bin/admincmd
export PATH;

SUFFIX=`date '+%d%m%Y%H%M%S'`;
OUTFOLDER="./LiveOpticsNetBackup"${SUFFIX};
CMP=gzip
PREFIX=quick

#
# Create the temp folder
#
echo "Making temporary folder for file output: $OUTFOLDER";
mkdir $OUTFOLDER;

#
# Check the disk space in the output folder.
#
DISK_SPACE_KB=0;

if [ $OS = 'SunOS' ]; then

	DISK_SPACE_KB=`df -b $OUTFOLDER | awk 'NR==2 { print $2 }'`;
	echo "$DISK_SPACE_KB KB available.";
	
elif [ $OS = 'Linux' ]; then

	DISK_SPACE_KB=`df -P $OUTFOLDER | awk '{if (NR!=1) { print $4 } }'`;
	DISK_SPACE_HUMAN=`df -Ph $OUTFOLDER | awk '{if (NR!=1) { print $4 } }'`;
	echo "$DISK_SPACE_HUMAN available.";

fi;

if [ "$DISK_SPACE_KB" -lt "4194304" ]; then
	echo "At least 4 GB of free disk space is recommended.";
	echo "Do you wish to proceed? (y/n) [n]:";
	read PROCEED;
	if [ "$PROCEED" = "y" -o "$PROCEED" = "Y" ]; then
		echo "Proceeding";
	else
		exit;
	fi;
fi;

#
# Write the local and the GMT times.
#
date > "$OUTFOLDER"/"$PREFIX"_date1.out
TZ=GMT
export TZ
date > "$OUTFOLDER"/"$PREFIX"_date.out

#
# Script version
#

echo $scriptVersion > "$OUTFOLDER"/"$PREFIX"_scriptVersion.out;


cat /usr/openv/netbackup/version > "$OUTFOLDER"/"$PREFIX"_version.out 

#
# Base date used for drawing the line on backups.
# 120 days is the default.
#
x=$(date +%s)
x=$((x - 120 * 24 * 60 * 60))
lastBackupDateString=`date --date @$x '+%m/%d/%Y'`

echo "Last backup of interest: "$lastBackupDateString;
echo $lastBackupDateString > "$OUTFOLDER"/"$PREFIX"_nbbasedate.out

#
# Grab a bunch of system settings.
#
set > "$OUTFOLDER"/"$PREFIX"_set.out
uname -a > "$OUTFOLDER"/"$PREFIX"_uname.out
hostname > "$OUTFOLDER"/"$PREFIX"_hostname.out

echo "Running command: bpconfig -L -U"
bpconfig -L -U  > "$OUTFOLDER"/"$PREFIX"_bpconfig.out 2> "$OUTFOLDER"/"$PREFIX"_bpconfig_err.out

echo "Running command: bpgetconfig -X"
bpgetconfig -X > "$OUTFOLDER"/"$PREFIX"_bpgetconfig.out 2> "$OUTFOLDER"/"$PREFIX"_bpgetconfig_err.out

echo "Running command: bpminlicense -list_keys -verbose"
bpminlicense -list_keys -verbose > "$OUTFOLDER"/"$PREFIX"_bpminlicense.out 2> "$OUTFOLDER"/"$PREFIX"_bpminlicense_err.out

echo "Running command: bpretlevel"
bpretlevel > "$OUTFOLDER"/"$PREFIX"_bpretlevel.out 2> "$OUTFOLDER"/"$PREFIX"_bpretlevel_err.out

echo "Running command: bpstulist -U -L"
bpstulist -U -L > "$OUTFOLDER"/"$PREFIX"_bpstulist.out 2> "$OUTFOLDER"/"$PREFIX"_bpstulist_err.out

echo "Running command: bpstulist -U -go -L"
bpstulist -U -go -L > "$OUTFOLDER"/"$PREFIX"_bpstulist_go.out 2> "$OUTFOLDER"/"$PREFIX"_bpstulist_go_err.out

echo "Running command: bpclient -All -L"
bpclient -All -L  > "$OUTFOLDER"/"$PREFIX"_bpclient.out 2> "$OUTFOLDER"/"$PREFIX"_bpclient_err.out

echo "Running command: bpplclients"
bpplclients > "$OUTFOLDER"/"$PREFIX"_bpplclients.out 2> "$OUTFOLDER"/"$PREFIX"_bpplclients_err.out

echo "Running command: bppllist -allpolicies -L -U" 
bppllist -allpolicies -L -U > "$OUTFOLDER"/"$PREFIX"_bppllist.out 2> "$OUTFOLDER"/"$PREFIX"_bppllist_err.out

echo "Running command: bppllist -allpolicies -l" 
bppllist -allpolicies -l > "$OUTFOLDER"/"$PREFIX"_bppllistl.out 2> "$OUTFOLDER"/"$PREFIX"_bppllistl_err.out

echo "Running command: bpimagelist -U -force -L -d $lastBackupDateString "
bpimagelist -U -force -L -d $lastBackupDateString 2>"$OUTFOLDER"/"$PREFIX"_bpimagelist_err.out | $CMP > "$OUTFOLDER"/"$PREFIX"_bpimagelist.out.gz

if [ -s "$OUTFOLDER"/"$PREFIX"_bpimagelist_err.out ] ; then
        # Error running bpimagelist. Attempting without -force option.
        mv "$OUTFOLDER"/"$PREFIX"_bpimagelist_err.out "$OUTFOLDER"/"$PREFIX"_bpimagelist_force_err.out
        echo "Running command: bpimagelist -U -L -d $lastBackupDateString "       
		bpimagelist -U -L -d $lastBackupDateString 2>"$OUTFOLDER"/"$PREFIX"_bpimagelist_err.out | $CMP > "$OUTFOLDER"/"$PREFIX"_bpimagelist.out.gz
fi

if [ ! -s "$OUTFOLDER"/"$PREFIX"_bpimagelist.out.gz ]; then
	echo "ERROR: bpimagelist command failed.";
	exit;
fi

echo "Running command: bpimmedia -U -force -l -d $lastBackupDateString "
bpimmedia -U -force -l -d $lastBackupDateString 2>"$OUTFOLDER"/"$PREFIX"_bpimmedial_err.out | $CMP > "$OUTFOLDER"/"$PREFIX"_bpimmedial.out.gz

if [ -s "$OUTFOLDER"/"$PREFIX"_bpimmedial_err.out ] ; then
		# Error running bpimmedia. Attempting without -force option. 
		mv "$OUTFOLDER"/"$PREFIX"_bpimmedial_err.out "$OUTFOLDER"/"$PREFIX"_bpimmedial_force_err.out
        echo "Running command: bpimmedia -U  -l -d $lastBackupDateString "
		bpimmedia -U -l -d $lastBackupDateString 2>"$OUTFOLDER"/"$PREFIX"_bpimmedial_err.out | $CMP > "$OUTFOLDER"/"$PREFIX"_bpimmedial.out.gz     
fi

if [ ! -s "$OUTFOLDER"/"$PREFIX"_bpimmedial.out.gz ]; then
	echo "ERROR: bpimmedia command failed.";
	exit;
fi

echo "Running command: bpmedialist -mlist -U -L"
bpmedialist -mlist -U -L | $CMP > "$OUTFOLDER"/"$PREFIX"_bpmedialist.out.gz

if [ ! -s "$OUTFOLDER"/"$PREFIX"_bpmedialist.out.gz ]; then
	echo "ERROR: bpmedialist -mlist -U -L failed.";
	exit;
fi

echo "Running command: bpmedialist -summary -U -L"
bpmedialist -summary -U -L > "$OUTFOLDER"/"$PREFIX"_bpmedialistsummary.out 2> "$OUTFOLDER"/"$PREFIX"_bpmedialistsummary_err.out

echo "Running command: vmquery -a "
vmquery -a > "$OUTFOLDER"/"$PREFIX"_vmquery.out 2> "$OUTFOLDER"/"$PREFIX"_vmquery_err.out

echo "Running command: tpconfig -dl"
tpconfig -dl > "$OUTFOLDER"/"$PREFIX"_tpconfig.out 2> "$OUTFOLDER"/"$PREFIX"_tpconfig_err.out

echo "Running command: tpconfig -dvirtualmachines"
tpconfig -dvirtualmachines > "$OUTFOLDER"/"$PREFIX"_virtualmachines_tpconfig.out 2> "$OUTFOLDER"/"$PREFIX"_tpconfig_virtualmachines_err.out

echo "Running command: tpconfig -ddiskarrays"
tpconfig -ddiskarrays > "$OUTFOLDER"/"$PREFIX"_tpconfig_diskarrays.out 2> "$OUTFOLDER"/"$PREFIX"_tpconfig_diskarrays_err.out

echo "Running command: vmglob -listall"
vmglob -listall > "$OUTFOLDER"/"$PREFIX"_vmglob.out 2> "$OUTFOLDER"/"$PREFIX"_vmglob_err.out

echo "Running command: vmpool -listall"
vmpool -listall > "$OUTFOLDER"/"$PREFIX"_vmpool.out 2> "$OUTFOLDER"/"$PREFIX"_vmpool_err.out

echo "Running command: vmoprcmd -d"
vmoprcmd -d > "$OUTFOLDER"/"$PREFIX"_vmoprcmd.out 2> "$OUTFOLDER"/"$PREFIX"_vmoprcmd_err.out

echo "Running command: vmoprcmd -devconfig -l"
vmoprcmd -devconfig -l > "$OUTFOLDER"/"$PREFIX"_vmoprcmd_devconfig.out 2> "$OUTFOLDER"/"$PREFIX"_vmoprcmd_devconfig_err.out

echo "Running command: nbdevquery -listdp -U"
nbdevquery -listdp -U >  "$OUTFOLDER"/"$PREFIX"_nbdevquery_listdp.out 2>  "$OUTFOLDER"/"$PREFIX"_nbdevquery_listdp_err.out

echo "Running command: nbdevquery -liststs -U" 
nbdevquery -liststs -U > "$OUTFOLDER"/"$PREFIX"_nbdevquery_liststs.out 2> "$OUTFOLDER"/"$PREFIX"_nbdevquery_liststs_err.out

echo "Running command: nbdevquery -listdv -U"
nbdevquery -listdv -stype IBM-ProtectTier -U >  "$OUTFOLDER"/"$PREFIX"_nbdevquery_listdv_pt.out 2>  "$OUTFOLDER"/"$PREFIX"_nbdevquery_listdv_pt_err.out

echo "Running command: nbdevquery -listdv -D"
nbdevquery -listdv -stype IBM-ProtectTier -D >  "$OUTFOLDER"/"$PREFIX"_nbdevquery_listdv_pt_d.out 2>  "$OUTFOLDER"/"$PREFIX"_nbdevquery_listdv_pt_d_err.out

echo "Running command: nbdevquery -listdv -U"
nbdevquery -listdv -stype DataDomain -U >  "$OUTFOLDER"/"$PREFIX"_nbdevquery_listdv_dd.out 2>  "$OUTFOLDER"/"$PREFIX"_nbdevquery_listdv_dd_err.out

echo "Running command: nbdevquery -listdv -D"
nbdevquery -listdv -stype DataDomain -D >  "$OUTFOLDER"/"$PREFIX"_nbdevquery_listdv_dd_d.out 2>  "$OUTFOLDER"/"$PREFIX"_nbdevquery_listdv_dd_d_err.out

echo "Running command: nbdevquery -listdv -U"
nbdevquery -listdv -stype PureDisk -U >  "$OUTFOLDER"/"$PREFIX"_nbdevquery_listdv_pd.out 2>  "$OUTFOLDER"/"$PREFIX"_nbdevquery_listdv_pd_err.out

echo "Running command: nbdevquery -listdv -D"
nbdevquery -listdv -stype PureDisk -D >  "$OUTFOLDER"/"$PREFIX"_nbdevquery_listdv_pd_d.out 2>  "$OUTFOLDER"/"$PREFIX"_nbdevquery_listdv_pd_d_err.out

echo "Running command: nbdevquery -listdv -U"
nbdevquery -listdv -stype AdvancedDisk -U >  "$OUTFOLDER"/"$PREFIX"_nbdevquery_listdv_ad.out 2>  "$OUTFOLDER"/"$PREFIX"_nbdevquery_listdv_ad_err.out

echo "Running command: nbdevquery -listdv -D"
nbdevquery -listdv -stype AdvancedDisk -D >  "$OUTFOLDER"/"$PREFIX"_nbdevquery_listdv_ad_d.out 2>  "$OUTFOLDER"/"$PREFIX"_nbdevquery_listdv_ad_d_err.out

echo "Running command: nbdevquery -listglobals -U"
nbdevquery -listglobals -U >  "$OUTFOLDER"/"$PREFIX"_nbdevquery_listglobals.out 2>  "$OUTFOLDER"/"$PREFIX"_nbdevquery_listglobals_err.out

echo "Running command: bpdbjobs -summary"
bpdbjobs -summary > "$OUTFOLDER"/"$PREFIX"_bpdbjobssummary.out 2> "$OUTFOLDER"/"$PREFIX"_bpdbjobssummary_err.out

echo "Running command: bpdbjobs -report -all_columns"
bpdbjobs -report -all_columns | $CMP > "$OUTFOLDER"/"$PREFIX"_bpdbjobs.out.gz

echo "Running command: nbftconfig -listclients -verbose"
nbftconfig -listclients -verbose | $CMP > "$OUTFOLDER"/"$PREFIX"_nbftconfig_listclients.out.gz

echo "Running command: nbftconfig -listservers -verbose"
nbftconfig -listservers -verbose | $CMP > "$OUTFOLDER"/"$PREFIX"_nbftconfig_listservers.out.gz

#
# Done.  Mark our final timestamp.
#
date > "$OUTFOLDER"/"$PREFIX"_date2.out

#
# TARGZ the files
#
echo "Packaging and compressing files into tar.gz file...";
PACKAGE="$PREFIX"_`hostname`.tar;

tar -cvf $PACKAGE "$OUTFOLDER"/"$PREFIX"_*out*;
$CMP $PACKAGE;

echo "Done.  Please upload the ${PACKAGE}.gz file to Live Optics.";

