<#
.SYNOPSIS
    Checks the current state for a list of specified tasks based on their name and prints the result
.DESCRIPTION
    Check for a list of tasks by their name and compare their current state with a plugin argument
    to determine the plugin output. States of tasks not matching the plugin argument will return
    [CRITICAL], while all other tasks will return [OK]. For not found tasks we will return [UNKNOWN]
    More Information on https://github.com/Icinga/icinga-powershell-plugins
.FUNCTIONALITY
    This plugin will let you check for scheduled tasks and their current state and compare it with the
    plugin argument to determine if they are fine or not
.EXAMPLE
    PS>Invoke-IcingaCheckScheduledTask -TaskName 'AutomaticBackup', 'Windows Backup Monitor' -State 'Ready';
    [OK] Check package "Scheduled Tasks"
    | 'automaticbackup_microsoftwindowswindowsbackup'=Ready;;Ready 'windows_backup_monitor_microsoftwindowswindowsbackup'=Ready;;Ready
.EXAMPLE
    PS>Invoke-IcingaCheckScheduledTask -TaskName 'AutomaticBackup', 'Windows Backup Monitor' -State 'Disabled';
    [CRITICAL] Check package "Scheduled Tasks" - [CRITICAL] AutomaticBackup (\Microsoft\Windows\WindowsBackup\), Windows Backup Monitor (\Microsoft\Windows\WindowsBackup\)
    \_ [CRITICAL] Check package "\Microsoft\Windows\WindowsBackup\"
        \_ [CRITICAL] AutomaticBackup (\Microsoft\Windows\WindowsBackup\): Value "Ready" is not matching threshold "Disabled"
        \_ [CRITICAL] Windows Backup Monitor (\Microsoft\Windows\WindowsBackup\): Value "Ready" is not matching threshold "Disabled"
    | 'automaticbackup_microsoftwindowswindowsbackup'=Ready;;Disabled 'windows_backup_monitor_microsoftwindowswindowsbackup'=Ready;;Disabled
.PARAMETER TaskName
   A list of tasks to check for. If your tasks contain spaces, wrap them around a ' to ensure they are
   properly handled as string
.PARAMETER State
   The state a task should currently have for the plugin to return [OK]
.PARAMETER NoPerfData
   Set this argument to not write any performance data
.PARAMETER Verbosity
    Changes the behavior of the plugin output which check states are printed:
    0 (default): Only service checks/packages with state not OK will be printed
    1: Only services with not OK will be printed including OK checks of affected check packages including Package config
    2: Everything will be printed regardless of the check state
    3: Identical to Verbose 2, but prints in addition the check package configuration e.g (All must be [OK])
.INPUTS
   System.String
.OUTPUTS
   System.String
.LINK
   https://github.com/Icinga/icinga-powershell-plugins
.NOTES
#>

function Invoke-IcingaCheckScheduledTask()
{
    param (
        [array]$TaskName,
        [ValidateSet('Unknown', 'Disabled', 'Queued', 'Ready', 'Running')]
        [array]$State       = @(),
        [switch]$NoPerfData,
        [ValidateSet(0, 1, 2, 3)]
        [int]$Verbosity     = 0
    );

    $TaskPackages  = New-IcingaCheckPackage -Name 'Scheduled Tasks' -OperatorAnd -Verbose $Verbosity;
    $Tasks         = Get-IcingaScheduledTask -TaskName $TaskName;

    foreach ($taskpath in $Tasks.Keys) {
        $TaskArray = $Tasks[$taskpath];

        $CheckPackage = New-IcingaCheckPackage -Name $taskpath -OperatorAnd -Verbose $Verbosity;

        foreach ($task in $TaskArray) {
            if ($taskpath -eq 'Unknown Tasks') {
                $CheckPackage.AddCheck(
                    (
                        New-IcingaCheck -Name ([string]::Format('{0}: Task not found', $task))
                    ).SetUnknown()
                )
            } else {
                $TaskCheck = New-IcingaCheck -Name ([string]::Format('{0} ({1})', $task.TaskName, $task.TaskPath)) -Value ($ProviderEnums.ScheduledTaskStatus[[string]$task.State]) -Translation $ProviderEnums.ScheduledTaskName;

                if ($State.Count -ne 0 -and $State -NotContains ([string]$task.State)) {
                    $TaskCheck.CritIfNotMatch($ProviderEnums.ScheduledTaskStatus[$State[0]]) | Out-Null;
                }
                $CheckPackage.AddCheck($TaskCheck);
            }
        }

        if ($CheckPackage.HasChecks()) {
            $TaskPackages.AddCheck($CheckPackage);
        }
    }

    return (New-IcingaCheckResult -Check $TaskPackages -NoPerfData $NoPerfData -Compile);
}
